/**
* \file: WaylandPointerListener.cpp
*
* \version: 0.3
*
* \release: $Name:$
*
* Handles pointer events coming from Wayland
*
* \component: Unified SPI
*
* \author: P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <string>
#include <linux/input.h>
#include <uspi_macros.h>
#include <uspi/ITouchFacade.h>
#include "WaylandPointerListener.h"

namespace adit { namespace uspi {
using std::to_string;

struct wl_pointer_listener WaylandPointerListener::mWlPointerListener =
{
    WaylandPointerListener::onEnter,
    WaylandPointerListener::onLeave,
    WaylandPointerListener::onMotion,
    WaylandPointerListener::onButton,
    WaylandPointerListener::onAxis,
    WaylandPointerListener::onFrame,
    WaylandPointerListener::onAxisSource,
    WaylandPointerListener::onAxisStop,
    WaylandPointerListener::onAxisDiscrete
};

WaylandPointerListener::WaylandPointerListener(ITouchFacadeCallbacks& inCallbacks, int inDisplayWidth,
        int inDisplayHeight)
: mCallbacks(inCallbacks), mXRelative(0), mYRelative(0), mDisplayWidth(inDisplayWidth), mDisplayHeight(inDisplayHeight),
  mButtons(0) { }

WaylandPointerListener::~WaylandPointerListener() { }

void WaylandPointerListener::setLateResolution(int inWidth, int inHeight)
{
    mDisplayWidth = inWidth;
    mDisplayHeight = inHeight;
}

/* Wayland pointer event handlers */
void WaylandPointerListener::onEnter(void* inMe, wl_pointer* inPointer, uint32_t inEventSerial, wl_surface* inWlSurface,
        wl_fixed_t inPixelX, wl_fixed_t inPixelY)
{
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inEventSerial);
    uspi_unused_variable(inWlSurface);

    auto me = static_cast<WaylandPointerListener*>(inMe);

    me->mCallbacks.onLogging(USPI_LOG_VERBOSE, "Pointer enter event on "
            + to_string(inPixelX) + " - " + to_string(inPixelY));

    me->mXRelative = (float)wl_fixed_to_double(inPixelX) / me->mDisplayWidth;
    me->mYRelative = (float)wl_fixed_to_double(inPixelY) / me->mDisplayHeight;
}

/* not interested in leave event */
void WaylandPointerListener::onLeave(void* inMe, wl_pointer* inPointer, uint32_t inEventSerial, wl_surface* inWlSurface)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inEventSerial);
    uspi_unused_variable(inWlSurface);
}

void WaylandPointerListener::onMotion(void* inMe, wl_pointer* inPointer, uint32_t inTimestamp, wl_fixed_t inPixelX,
        wl_fixed_t inPixelY)
{
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inTimestamp);

    auto me = static_cast<WaylandPointerListener*>(inMe);

    me->mCallbacks.onLogging(USPI_LOG_VERBOSE, "Pointer motion event on "
            + to_string(inPixelX) + " - " + to_string(inPixelY) + " Buttons: " + to_string(me->mButtons));

    me->mXRelative = (float)wl_fixed_to_double(inPixelX) / me->mDisplayWidth;
    me->mYRelative = (float)wl_fixed_to_double(inPixelY) / me->mDisplayHeight;

    if(me->mButtons)
    {
        TouchEvent currentTouchEvent;
        currentTouchEvent.fingerId = 0;
        /* Wayland does not detect pressure */
        currentTouchEvent.pressure = 1;
        currentTouchEvent.xRelative = me->mXRelative;
        currentTouchEvent.yRelative = me->mYRelative;
        currentTouchEvent.eventType = TouchEvent::move;

        me->mCallbacks.onTouch(currentTouchEvent);
    }

}

void WaylandPointerListener::onButton(void* inMe, wl_pointer* inPointer, uint32_t inEventSerial, uint32_t inTimestamp,
        uint32_t inButton, uint32_t inButtonState)
{
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inEventSerial);
    uspi_unused_variable(inTimestamp);

    auto me = static_cast<WaylandPointerListener*>(inMe);

    me->mCallbacks.onLogging(USPI_LOG_VERBOSE, "Pointer button event Button: "
            + to_string(inButton) + " State: " + to_string(inButtonState));

    uint8_t button = 0;
    switch (inButton)
    {
    case BTN_TOUCH:
        button |= 1;
        break;
    case BTN_LEFT:
        button |= 1 << 1;
        break;
    case BTN_RIGHT:
        button |= 1 << 2;
        break;
    case BTN_MIDDLE:
        button |= 1 << 3;
        break;
    }

    bool previously_pressed = (me->mButtons != 0);

    if (inButtonState == 0)
    {
        /* remove all up bits */
        me->mButtons &= ~button;
    }
    else
    {
        /* add down bits */
        me->mButtons |= button;
    }

    if (!previously_pressed && me->mButtons != 0)
    {
        /* first button was pressed */
        TouchEvent currentTouchEvent;
        currentTouchEvent.fingerId = 0;
        /* Wayland does not detect pressure */
        currentTouchEvent.pressure = 1;
        currentTouchEvent.xRelative = me->mXRelative;
        currentTouchEvent.yRelative = me->mYRelative;
        currentTouchEvent.eventType = TouchEvent::down;

        me->mCallbacks.onTouch(currentTouchEvent);
    }
    else if (previously_pressed && me->mButtons == 0)
    {
        /* last button was released */
        TouchEvent currentTouchEvent;
        currentTouchEvent.fingerId = 0;
        /* Wayland does not detect pressure */
        currentTouchEvent.pressure = 0;
        currentTouchEvent.xRelative = me->mXRelative;
        currentTouchEvent.yRelative = me->mYRelative;
        currentTouchEvent.eventType = TouchEvent::up;

        me->mCallbacks.onTouch(currentTouchEvent);
    }
}

/* not interested in axis event */
void WaylandPointerListener::onAxis(void* inMe, wl_pointer* inPointer, uint32_t inTimestamp, uint32_t inAxis,
        wl_fixed_t inValue)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inTimestamp);
    uspi_unused_variable(inAxis);
    uspi_unused_variable(inValue);
}

/* not interested in frame event */
void WaylandPointerListener::onFrame(void* inMe, wl_pointer* inPointer)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inPointer);
}

/* not interested in axis source event */
void WaylandPointerListener::onAxisSource(void* inMe, wl_pointer* inPointer, uint32_t inAxisSource)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inAxisSource);
}

/* not interested in axis stop event */
void WaylandPointerListener::onAxisStop(void* inMe, wl_pointer* inPointer, uint32_t inTimestamp, uint32_t inAxis)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inTimestamp);
    uspi_unused_variable(inAxis);
}

/* not interested in axis discrete event */
void WaylandPointerListener::onAxisDiscrete(void* inMe, wl_pointer* inPointer, uint32_t inAxis, int32_t inDiscrete)
{
    uspi_unused_variable(inMe);
    uspi_unused_variable(inPointer);
    uspi_unused_variable(inAxis);
    uspi_unused_variable(inDiscrete);
}

} } /* namespace adit { namespace uspi { */
